/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2004.
-------------------------------------------------------------------------
$Id: AnimActionManager.cpp,v 1.1 2008/11/11 15:51:32 PauloZaffari Exp wwwrun $
$DateTime$
Description:  This is the source file for the module AnimActionManager in
Sanbox. Its purpose is to serve as a facade for the real AnimAction
manager thus allowing implementation to vary independently of the interface
used to represent it.
-------------------------------------------------------------------------
History:
- 11:11:2008   15:51 : Created by Paulo Zaffari

*************************************************************************/

#include "stdafx.h"
#include "EditorAnimActionManager.h"

#include "ICryAnimation.h"
#include "IAnimAction.h"

//////////////////////////////////////////////////////////////////////////
CEditorAnimActionManager::CEditorAnimActionManager():
m_piCharacter(NULL),
m_piAnimActionManager(NULL)
{
	ISystem*														piSystem(NULL);
	ICharacterManager*									piAnimationSystem(NULL);
	IAnimActionManager*									piAnimActionManager(NULL);

	piSystem=GetISystem();
	if (!piSystem)
	{
		// No system, we can't proceed.
		return;
	}

	piAnimationSystem = piSystem->GetIAnimationSystem();
	if (!piAnimationSystem)
	{
		// No animation system, we can't proceed.
		return;
	}

	piAnimActionManager=piAnimationSystem->GetIAnimActionManager();
	if (!piAnimActionManager)
	{
		// We failed to get the interface to AnimAction manager.
		return;
	}

	m_piAnimActionManager=piAnimActionManager;
}
//////////////////////////////////////////////////////////////////////////
CEditorAnimActionManager::~CEditorAnimActionManager()
{

}
//////////////////////////////////////////////////////////////////////////
CEditorAnimActionManager& CEditorAnimActionManager::GetAnimActionManager()
{
	static CEditorAnimActionManager oAnimActionManager;
	return oAnimActionManager;
}
//////////////////////////////////////////////////////////////////////////
void CEditorAnimActionManager::GetAnimActionClasses(TDAnimActions& rcpiAnimActionClasses)
{
	size_t															nNumberOfClasses(0);
	size_t															nCurrentClass(0);

	if (!m_piAnimActionManager)
	{
		return;
	}

	nNumberOfClasses=m_piAnimActionManager->GetAAClassCount();
	rcpiAnimActionClasses.resize(0);
	rcpiAnimActionClasses.reserve(nNumberOfClasses);

	for (nCurrentClass=0;nCurrentClass<nNumberOfClasses;++nCurrentClass)
	{
		rcpiAnimActionClasses.push_back(m_piAnimActionManager->GetAAClass(nCurrentClass));
	}	
}
//////////////////////////////////////////////////////////////////////////
void CEditorAnimActionManager::GetAnimActionPrototypes(TDAnimActions& rcpiAnimActionPrototypes)
{
	rcpiAnimActionPrototypes=m_cpiAnimActionPrototypes;
}
//////////////////////////////////////////////////////////////////////////
IAnimActionClass* CEditorAnimActionManager::CreateAnimActionPrototype(IAnimActionClass* piAnimActionClass,string& strAnimActionName,string& strAnimActionFilename)
{
	string																								strSaveFilename;
	IAnimActionClass*																			piNewAnimationPrototype(NULL);
	const SAnimActionPropertyDesc*												astPropertiesDesc(NULL);
	size_t																								nCurrentPropertyIndex(0);

	size_t																								nAnimActionIndex(-1);

	XmlNodeRef																						oCurrentNode;
	XmlNodeRef																						oParentNode;
	XmlNodeRef																						oRootNode;

	if (!m_piAnimActionManager)
	{
		return NULL;
	}

	strSaveFilename=m_strBaseDirectory;
	strSaveFilename+=strAnimActionFilename;

	nAnimActionIndex=GetPrototypeIndex(strSaveFilename);

	oRootNode=GetISystem()->CreateXmlNode("AnimAction");
	oRootNode->setAttr("class",	piAnimActionClass->GetAAClassName());

	oParentNode=oRootNode;
	oCurrentNode=oRootNode;

	astPropertiesDesc=piAnimActionClass->GetPropertiesDesc();
	// If we have properties, let us write them to the XML file.
	if (astPropertiesDesc)
	{
		// For all valid descriptors (before the null descriptor is found)...
		for (
					nCurrentPropertyIndex=0;
					astPropertiesDesc[nCurrentPropertyIndex].propertyType!=eAAPT_null;
					++nCurrentPropertyIndex
				)
		{
			// We check for the kind of property we have...
			// ...creating the appropriate variable match and setting the correct
			// data type for it.
			oCurrentNode=oParentNode->createNode("Param");
			oParentNode->addChild(oCurrentNode);

			oCurrentNode->setAttr("id",astPropertiesDesc[nCurrentPropertyIndex].propertyName);
			oCurrentNode->setAttr("value",astPropertiesDesc[nCurrentPropertyIndex].defaultValue);
		}
	}

	oRootNode->saveToFile(strSaveFilename.c_str());

	piNewAnimationPrototype=m_piAnimActionManager->CreatePrototypeFromXml(strAnimActionFilename.c_str(),oRootNode);
	if (piNewAnimationPrototype)
	{
		if (nAnimActionIndex!=-1)
		{
			delete(m_cpiAnimActionPrototypes[nAnimActionIndex]);
			m_cpiAnimActionPrototypes[nAnimActionIndex]=piNewAnimationPrototype;
			m_cstrAnimActionFilenames[nAnimActionIndex]=strSaveFilename;
		}
		else
		{
			m_cpiAnimActionPrototypes.push_back(piNewAnimationPrototype);
			m_cstrAnimActionFilenames.push_back(strSaveFilename);
		}
	}

	return piNewAnimationPrototype;
}
//////////////////////////////////////////////////////////////////////////
void		CEditorAnimActionManager::SetBaseDirectory(const string& strBaseDirectory)
{
	m_strBaseDirectory=strBaseDirectory;

	LoadAllAnimActions();
}
//////////////////////////////////////////////////////////////////////////
string&    CEditorAnimActionManager::GetAnimActionFilename(IAnimActionClass* piAnimAction,string& strAnimActionFileName)
{
	size_t nAnimActionId(0);

	nAnimActionId=GetPrototypeIndex(piAnimAction);
	if (nAnimActionId<m_cstrAnimActionFilenames.size())
	{
		strAnimActionFileName=m_cstrAnimActionFilenames[nAnimActionId];
	}
	return strAnimActionFileName;
}
//////////////////////////////////////////////////////////////////////////
bool	CEditorAnimActionManager::SavePrototype(IAnimActionClass* piAnimActionPrototype)
{
	string																								strSaveFilename;
	IAnimActionClass*																			piNewAnimationPrototype(NULL);
	const SAnimActionPropertyDesc*												astPropertiesDesc(NULL);
	size_t																								nCurrentPropertyIndex(0);
	size_t																								nPrototypeIndex(0);

	XmlNodeRef																						oCurrentNode;
	XmlNodeRef																						oParentNode;
	XmlNodeRef																						oRootNode;
	bool																									boSaveOperationSuccess(false);

	if (!m_piAnimActionManager)
	{
		return false;
	}

	nPrototypeIndex=GetPrototypeIndex(piAnimActionPrototype);
	if (nPrototypeIndex==-1)
	{
		return false;
	}

	oRootNode=GetISystem()->CreateXmlNode("AnimAction");
	oRootNode->setAttr("class",piAnimActionPrototype->GetAAClassName());

	oParentNode=oRootNode;
	oCurrentNode=oRootNode;

	astPropertiesDesc=piAnimActionPrototype->GetPropertiesDesc();
	TAnimActionPropertyValues& rcPropertyValues=piAnimActionPrototype->GetProperties();

	// If we have properties, let us write them to the XML file.
	if (astPropertiesDesc)
	{
		// For all valid descriptors (before the null descriptor is found)...
		for (
			nCurrentPropertyIndex=0;
			astPropertiesDesc[nCurrentPropertyIndex].propertyType!=eAAPT_null;
		++nCurrentPropertyIndex
			)
		{
			// We check for the kind of property we have...
			// ...creating the appropriate variable match and setting the correct
			// data type for it.
			oCurrentNode=oParentNode->createNode("Param");
			oParentNode->addChild(oCurrentNode);

			oCurrentNode->setAttr("id",astPropertiesDesc[nCurrentPropertyIndex].propertyName);
			oCurrentNode->setAttr("value",rcPropertyValues[nCurrentPropertyIndex].GetAsString(&astPropertiesDesc[nCurrentPropertyIndex]));
		}
	}

	strSaveFilename=m_cstrAnimActionFilenames[nPrototypeIndex];
	boSaveOperationSuccess=oRootNode->saveToFile(strSaveFilename.c_str());

	return true;
}
//////////////////////////////////////////////////////////////////////////
bool	CEditorAnimActionManager::DeletePrototype(IAnimActionClass*& piAnimActionPrototype)
{
	size_t	nPrototypeIndex(0);
	BOOL		bnReturnValue(TRUE);

	string	strDeleteFilename;

	nPrototypeIndex=GetPrototypeIndex(piAnimActionPrototype);
	m_cpiAnimActionPrototypes.erase(m_cpiAnimActionPrototypes.begin()+nPrototypeIndex);

	strDeleteFilename+=m_cstrAnimActionFilenames[nPrototypeIndex];
	m_cstrAnimActionFilenames.erase(m_cstrAnimActionFilenames.begin()+nPrototypeIndex);
	
	bnReturnValue=CFileUtil::DeleteFile(strDeleteFilename.c_str());

	delete(piAnimActionPrototype);
	piAnimActionPrototype=NULL;
	
	return (bnReturnValue!=FALSE);
}
//////////////////////////////////////////////////////////////////////////
bool	CEditorAnimActionManager::RenamePrototype(IAnimActionClass* piAnimActionPrototype,const string& strNewAnimActionFilename)
{
	size_t	nPrototypeIndex(0);
	string	strOldFilename;
	BOOL		bnReturnValue(TRUE);

	nPrototypeIndex=GetPrototypeIndex(piAnimActionPrototype);

	if (nPrototypeIndex>=m_cstrAnimActionFilenames.size())
	{
		return false;
	}

	strOldFilename=m_cstrAnimActionFilenames[nPrototypeIndex];
	m_cstrAnimActionFilenames[nPrototypeIndex]=strNewAnimActionFilename;
	
	bnReturnValue=rename(strOldFilename.c_str(),strNewAnimActionFilename.c_str());

	return bnReturnValue;
}
//////////////////////////////////////////////////////////////////////////
bool		CEditorAnimActionManager::MovePrototype(IAnimActionClass* piAnimActionPrototype,const string& strDirectoryPath)
{
	size_t		nPrototypeIndex(0);
	bool			boReturnValue(false);

	CString		strDriveLetter;
	CString		strPath;
	CString		strFilename;
	CString		strExtension;
	CString		strTargetFilename;

	CString		strOriginalFilename;
	
	nPrototypeIndex=GetPrototypeIndex(piAnimActionPrototype);

	if (nPrototypeIndex>=m_cstrAnimActionFilenames.size())
	{
		return false;
	}
	strOriginalFilename=m_cstrAnimActionFilenames[nPrototypeIndex].c_str();
	
	Path::SplitPath((CString)strDirectoryPath.c_str(),strDriveLetter,strPath,strFilename,strExtension);

	strTargetFilename=strDriveLetter;
	strTargetFilename+=strPath;
	strTargetFilename=Path::AddBackslash(strTargetFilename);

	Path::SplitPath(strOriginalFilename,strDriveLetter,strPath,strFilename,strExtension);

	strTargetFilename+=strFilename;
	strTargetFilename+=strExtension;

	boReturnValue=(CFileUtil::MoveFile(strOriginalFilename,strTargetFilename)==CFileUtil::ETREECOPYOK);
	m_cstrAnimActionFilenames[nPrototypeIndex]=strTargetFilename.GetBuffer();

	return boReturnValue;
}
//////////////////////////////////////////////////////////////////////////
void	CEditorAnimActionManager::ReloadAllAnimActions()
{
	UnloadAllAnimAction();
	LoadAllAnimActions();
}
//////////////////////////////////////////////////////////////////////////
IAnimActionClass* CEditorAnimActionManager::LoadAnimAction(string& strAnimActionFilename)
{
	CFileUtil::FileArray										cFiles;
	size_t																	nTotalNumberOfAnimActions(0);
	size_t																	nCurrentAnimActionBeingLoaded(0);
	size_t																	nCurrentAnimActionIndex(0);
	IAnimActionClass*												piCurrentPrototype(NULL);
	XmlNodeRef															oCurrentXMLNode;
	string																	strFullPath;

	if (!m_piAnimActionManager)
	{
		return NULL;
	}

	strFullPath=strAnimActionFilename;
	oCurrentXMLNode=GetISystem()->LoadXmlFile(strFullPath);
	piCurrentPrototype=m_piAnimActionManager->CreatePrototypeFromXml(strAnimActionFilename.c_str(),oCurrentXMLNode);

	// We won't add to our AnimAction database elements that failed to load.
	if (!piCurrentPrototype)
	{
		return NULL;
	}

	nCurrentAnimActionIndex=GetPrototypeIndex(strAnimActionFilename.c_str());
	if (nCurrentAnimActionIndex!=-1)
	{
		delete(m_cpiAnimActionPrototypes[nCurrentAnimActionIndex]);
		m_cpiAnimActionPrototypes[nCurrentAnimActionIndex]=piCurrentPrototype;
	}
	else
	{
		m_cpiAnimActionPrototypes.push_back(piCurrentPrototype);
		// We are saving the path relative to main directory.
		m_cstrAnimActionFilenames.push_back(strFullPath);
	}

	return piCurrentPrototype;
}
//////////////////////////////////////////////////////////////////////////
void		CEditorAnimActionManager::LoadAllAnimActions()
{
	CFileUtil::FileArray										cFiles;
	size_t																	nTotalNumberOfAnimActions(0);
	size_t																	nCurrentAnimActionBeingLoaded(0);
	IAnimActionClass*												piCurrentPrototype(NULL);
	XmlNodeRef															oCurrentXMLNode;
	string																	strFullPath;

	if (!m_piAnimActionManager)
	{
		return;
	}

	UnloadAllAnimAction();

	CFileUtil::ScanDirectory(m_strBaseDirectory.c_str(),"*.CAA",cFiles,true);

	nTotalNumberOfAnimActions=cFiles.size();
	for (	
				nCurrentAnimActionBeingLoaded=0;
				nCurrentAnimActionBeingLoaded<nTotalNumberOfAnimActions;
				++nCurrentAnimActionBeingLoaded
			)
				{
					strFullPath=m_strBaseDirectory;
					strFullPath+=cFiles[nCurrentAnimActionBeingLoaded].filename.GetBuffer();
					oCurrentXMLNode=GetISystem()->LoadXmlFile(strFullPath);
					piCurrentPrototype=m_piAnimActionManager->CreatePrototypeFromXml(cFiles[nCurrentAnimActionBeingLoaded].filename.GetBuffer(),oCurrentXMLNode);
					// We won't add to our AnimAction database elements that failed to load.
					if (!piCurrentPrototype)
					{
						continue;
					}
					m_cpiAnimActionPrototypes.push_back(piCurrentPrototype);
					// We are saving the path relative to main directory.
					m_cstrAnimActionFilenames.push_back(strFullPath);
				}
}
//////////////////////////////////////////////////////////////////////////
void		CEditorAnimActionManager::UnloadAllAnimAction()
{
	size_t	nCurrentPrototype(0);
	size_t	nTotalPrototypes(0);

	if (!m_piAnimActionManager)
	{
		return;
	}

	nTotalPrototypes=m_cpiAnimActionPrototypes.size();
	for (nCurrentPrototype=0;nCurrentPrototype<nTotalPrototypes;++nCurrentPrototype)
	{
		// Eventually, calling the manager might be needed as well.
		delete (m_cpiAnimActionPrototypes[nCurrentPrototype]);
	}

	m_cpiAnimActionPrototypes.clear();
	m_cstrAnimActionFilenames.clear();
}
//////////////////////////////////////////////////////////////////////////
size_t  CEditorAnimActionManager::GetPrototypeIndex(IAnimActionClass* piAnimAction)
{
	size_t															nNumberOfAnimations(0);
	size_t															nCurrentAnimation(0);

	nNumberOfAnimations=m_cpiAnimActionPrototypes.size();
	for (	
				nCurrentAnimation=0;
				nCurrentAnimation<nNumberOfAnimations;
				++nCurrentAnimation
			)
	{
		if (piAnimAction==m_cpiAnimActionPrototypes[nCurrentAnimation])
		{
			return nCurrentAnimation;
		}		
	}

	return -1;
}
//////////////////////////////////////////////////////////////////////////
size_t  CEditorAnimActionManager::GetPrototypeIndex(const string& crstrAnimactionFilename)
{
	size_t															nNumberOfAnimations(0);
	size_t															nCurrentAnimation(0);

	nNumberOfAnimations=m_cpiAnimActionPrototypes.size();
	for (	
		nCurrentAnimation=0;
		nCurrentAnimation<nNumberOfAnimations;
	++nCurrentAnimation
		)
	{
		if (crstrAnimactionFilename==m_cstrAnimActionFilenames[nCurrentAnimation])
		{
			return nCurrentAnimation;
		}		
	}

	return -1;
}
//////////////////////////////////////////////////////////////////////////
